import sys

import numpy as np
import TransientImage
import scipy.interpolate
import matplotlib.pyplot as plt
import json

def ApplyPhaseFunction(ti, phaseFile):
	phase_orig = np.fromfile(phaseFile, dtype=np.float32)
	phase_orig_t = np.tile(phase_orig, 5)
	l = phase_orig_t.shape[0]
	
	# remap this to 0..size-1
	size=1600
	phase = scipy.interpolate.interp1d(np.arange(0, l, 1)*((size-1)/(l-1)), phase_orig_t, kind="quadratic")(np.arange(0, size, 1))
	
	# now fold the image with the phase function
	return np.dot(ti.data, phase)/np.max(np.abs(phase))

def ApplyNoiseToTi(ti, photonScale=1000000, quantumEfficiency=0.7, darkNoise=2.3):
	photons = np.random.poisson(np.round(ti.data*photonScale), ti.data.shape) # shot noise
	electrons = np.round(photons * quantumEfficiency)
	e_out = electrons + np.round(np.random.normal(loc=0, scale=darkNoise, size=electrons.shape))
	e_out[e_out<0] = 0 # avoid negative values
	
	ti.data = e_out

def ApplyNoiseToFile(filename, photonScale=1000000, quantumEfficiency=0.7, darkNoise=2.3, outputFilename=None):
	if not filename.endswith('.ti'):
		raise Exception("Please provide transient image file")
	ti = TransientImage.TransientImage(filename)
	ApplyNoiseToTi(ti)
	if outputFilename is None:
		outputFilename = filename[:-3]+"-phot_{}-quat_{}-dark_{}.ti".format(photonScale, quantumEfficiency, darkNoise)
	
	# write noise parameters to file
	properties = json.loads(ti.imageProperties)
	if 'ShotNoise' in properties:
		raise Exception("it is not meaningful to apply ShotNoise twice.")
	properties['ShotNoise'] = {'photonScale' : photonScale, 'quantumEfficiency' : quantumEfficiency, 'darkNoise' : darkNoise}
	ti.imageProperties = json.dumps(properties, sort_keys=True, indent=4)
	
	ti.SaveFile(outputFilename)


##############################################################


# example usage:
# ApplyNoiseToFile("Hammer.ti", photonScale=100000000, quantumEfficiency=0.7, darkNoise=2.3)
